// -------------------------------------------------------------------
// This is a library for Monochrome OLEDs based on SSD1306 drivers
// It supports the 128 x 64 and the 64 x 48 displays
// You can also configure the orientation of the display
// To coonfigure the displaytype and orention just change the defines
// in the top of this file for the memory buffer.
// Using this type of display will cost a lot of memory for the buffer
// Written by: Hein Pragt  in 2017   www.heinpragt.com 
// This code is free of any licence, use it the way you like :-)
// -------------------------------------------------------------------

#ifdef __AVR__
#include <avr/pgmspace.h>
#endif

#include <stdlib.h>
#include <Wire.h>
#include "SSD1306.h"

  
// size of display for memory buffer
// change if you want to save mem using a 64x48 display
#define SSD1306_LCDWIDTH 128
#define SSD1306_LCDHEIGHT 64

#define BUFFERSIZE  (SSD1306_LCDHEIGHT * SSD1306_LCDWIDTH / 8)

// Memory buffer for the display

static uint8_t buffer[BUFFERSIZE];


// Standard ASCII 5x7 font

static const unsigned char font[] PROGMEM = {
	0x00, 0x00, 0x00, 0x00, 0x00,
	0x3E, 0x5B, 0x4F, 0x5B, 0x3E,
	0x3E, 0x6B, 0x4F, 0x6B, 0x3E,
	0x1C, 0x3E, 0x7C, 0x3E, 0x1C,
	0x18, 0x3C, 0x7E, 0x3C, 0x18,
	0x1C, 0x57, 0x7D, 0x57, 0x1C,
	0x1C, 0x5E, 0x7F, 0x5E, 0x1C,
	0x00, 0x18, 0x3C, 0x18, 0x00,
	0xFF, 0xE7, 0xC3, 0xE7, 0xFF,
	0x00, 0x18, 0x24, 0x18, 0x00,
	0xFF, 0xE7, 0xDB, 0xE7, 0xFF,
	0x30, 0x48, 0x3A, 0x06, 0x0E,
	0x26, 0x29, 0x79, 0x29, 0x26,
	0x40, 0x7F, 0x05, 0x05, 0x07,
	0x40, 0x7F, 0x05, 0x25, 0x3F,
	0x5A, 0x3C, 0xE7, 0x3C, 0x5A,
	0x7F, 0x3E, 0x1C, 0x1C, 0x08,
	0x08, 0x1C, 0x1C, 0x3E, 0x7F,
	0x14, 0x22, 0x7F, 0x22, 0x14,
	0x5F, 0x5F, 0x00, 0x5F, 0x5F,
	0x06, 0x09, 0x7F, 0x01, 0x7F,
	0x00, 0x66, 0x89, 0x95, 0x6A,
	0x60, 0x60, 0x60, 0x60, 0x60,
	0x94, 0xA2, 0xFF, 0xA2, 0x94,
	0x08, 0x04, 0x7E, 0x04, 0x08,
	0x10, 0x20, 0x7E, 0x20, 0x10,
	0x08, 0x08, 0x2A, 0x1C, 0x08,
	0x08, 0x1C, 0x2A, 0x08, 0x08,
	0x1E, 0x10, 0x10, 0x10, 0x10,
	0x0C, 0x1E, 0x0C, 0x1E, 0x0C,
	0x30, 0x38, 0x3E, 0x38, 0x30,
	0x06, 0x0E, 0x3E, 0x0E, 0x06,
	0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x5F, 0x00, 0x00,
	0x00, 0x07, 0x00, 0x07, 0x00,
	0x14, 0x7F, 0x14, 0x7F, 0x14,
	0x24, 0x2A, 0x7F, 0x2A, 0x12,
	0x23, 0x13, 0x08, 0x64, 0x62,
	0x36, 0x49, 0x56, 0x20, 0x50,
	0x00, 0x08, 0x07, 0x03, 0x00,
	0x00, 0x1C, 0x22, 0x41, 0x00,
	0x00, 0x41, 0x22, 0x1C, 0x00,
	0x2A, 0x1C, 0x7F, 0x1C, 0x2A,
	0x08, 0x08, 0x3E, 0x08, 0x08,
	0x00, 0x80, 0x70, 0x30, 0x00,
	0x08, 0x08, 0x08, 0x08, 0x08,
	0x00, 0x00, 0x60, 0x60, 0x00,
	0x20, 0x10, 0x08, 0x04, 0x02,
	0x3E, 0x51, 0x49, 0x45, 0x3E,
	0x00, 0x42, 0x7F, 0x40, 0x00,
	0x72, 0x49, 0x49, 0x49, 0x46,
	0x21, 0x41, 0x49, 0x4D, 0x33,
	0x18, 0x14, 0x12, 0x7F, 0x10,
	0x27, 0x45, 0x45, 0x45, 0x39,
	0x3C, 0x4A, 0x49, 0x49, 0x31,
	0x41, 0x21, 0x11, 0x09, 0x07,
	0x36, 0x49, 0x49, 0x49, 0x36,
	0x46, 0x49, 0x49, 0x29, 0x1E,
	0x00, 0x00, 0x14, 0x00, 0x00,
	0x00, 0x40, 0x34, 0x00, 0x00,
	0x00, 0x08, 0x14, 0x22, 0x41,
	0x14, 0x14, 0x14, 0x14, 0x14,
	0x00, 0x41, 0x22, 0x14, 0x08,
	0x02, 0x01, 0x59, 0x09, 0x06,
	0x3E, 0x41, 0x5D, 0x59, 0x4E,
	0x7C, 0x12, 0x11, 0x12, 0x7C,
	0x7F, 0x49, 0x49, 0x49, 0x36,
	0x3E, 0x41, 0x41, 0x41, 0x22,
	0x7F, 0x41, 0x41, 0x41, 0x3E,
	0x7F, 0x49, 0x49, 0x49, 0x41,
	0x7F, 0x09, 0x09, 0x09, 0x01,
	0x3E, 0x41, 0x41, 0x51, 0x73,
	0x7F, 0x08, 0x08, 0x08, 0x7F,
	0x00, 0x41, 0x7F, 0x41, 0x00,
	0x20, 0x40, 0x41, 0x3F, 0x01,
	0x7F, 0x08, 0x14, 0x22, 0x41,
	0x7F, 0x40, 0x40, 0x40, 0x40,
	0x7F, 0x02, 0x1C, 0x02, 0x7F,
	0x7F, 0x04, 0x08, 0x10, 0x7F,
	0x3E, 0x41, 0x41, 0x41, 0x3E,
	0x7F, 0x09, 0x09, 0x09, 0x06,
	0x3E, 0x41, 0x51, 0x21, 0x5E,
	0x7F, 0x09, 0x19, 0x29, 0x46,
	0x26, 0x49, 0x49, 0x49, 0x32,
	0x03, 0x01, 0x7F, 0x01, 0x03,
	0x3F, 0x40, 0x40, 0x40, 0x3F,
	0x1F, 0x20, 0x40, 0x20, 0x1F,
	0x3F, 0x40, 0x38, 0x40, 0x3F,
	0x63, 0x14, 0x08, 0x14, 0x63,
	0x03, 0x04, 0x78, 0x04, 0x03,
	0x61, 0x59, 0x49, 0x4D, 0x43,
	0x00, 0x7F, 0x41, 0x41, 0x41,
	0x02, 0x04, 0x08, 0x10, 0x20,
	0x00, 0x41, 0x41, 0x41, 0x7F,
	0x04, 0x02, 0x01, 0x02, 0x04,
	0x40, 0x40, 0x40, 0x40, 0x40,
	0x00, 0x03, 0x07, 0x08, 0x00,
	0x20, 0x54, 0x54, 0x78, 0x40,
	0x7F, 0x28, 0x44, 0x44, 0x38,
	0x38, 0x44, 0x44, 0x44, 0x28,
	0x38, 0x44, 0x44, 0x28, 0x7F,
	0x38, 0x54, 0x54, 0x54, 0x18,
	0x00, 0x08, 0x7E, 0x09, 0x02,
	0x18, 0xA4, 0xA4, 0x9C, 0x78,
	0x7F, 0x08, 0x04, 0x04, 0x78,
	0x00, 0x44, 0x7D, 0x40, 0x00,
	0x20, 0x40, 0x40, 0x3D, 0x00,
	0x7F, 0x10, 0x28, 0x44, 0x00,
	0x00, 0x41, 0x7F, 0x40, 0x00,
	0x7C, 0x04, 0x78, 0x04, 0x78,
	0x7C, 0x08, 0x04, 0x04, 0x78,
	0x38, 0x44, 0x44, 0x44, 0x38,
	0xFC, 0x18, 0x24, 0x24, 0x18,
	0x18, 0x24, 0x24, 0x18, 0xFC,
	0x7C, 0x08, 0x04, 0x04, 0x08,
	0x48, 0x54, 0x54, 0x54, 0x24,
	0x04, 0x04, 0x3F, 0x44, 0x24,
	0x3C, 0x40, 0x40, 0x20, 0x7C,
	0x1C, 0x20, 0x40, 0x20, 0x1C,
	0x3C, 0x40, 0x30, 0x40, 0x3C,
	0x44, 0x28, 0x10, 0x28, 0x44,
	0x4C, 0x90, 0x90, 0x90, 0x7C,
	0x44, 0x64, 0x54, 0x4C, 0x44,
	0x00, 0x08, 0x36, 0x41, 0x00,
	0x00, 0x00, 0x77, 0x00, 0x00,
	0x00, 0x41, 0x36, 0x08, 0x00,
	0x02, 0x01, 0x02, 0x04, 0x02,
	0x3C, 0x26, 0x23, 0x26, 0x3C,
	0x1E, 0xA1, 0xA1, 0x61, 0x12,
	0x3A, 0x40, 0x40, 0x20, 0x7A,
	0x38, 0x54, 0x54, 0x55, 0x59,
	0x21, 0x55, 0x55, 0x79, 0x41,
	0x22, 0x54, 0x54, 0x78, 0x42, 
	0x21, 0x55, 0x54, 0x78, 0x40,
	0x20, 0x54, 0x55, 0x79, 0x40,
	0x0C, 0x1E, 0x52, 0x72, 0x12,
	0x39, 0x55, 0x55, 0x55, 0x59,
	0x39, 0x54, 0x54, 0x54, 0x59,
	0x39, 0x55, 0x54, 0x54, 0x58,
	0x00, 0x00, 0x45, 0x7C, 0x41,
	0x00, 0x02, 0x45, 0x7D, 0x42,
	0x00, 0x01, 0x45, 0x7C, 0x40,
	0x7D, 0x12, 0x11, 0x12, 0x7D, 
	0xF0, 0x28, 0x25, 0x28, 0xF0,
	0x7C, 0x54, 0x55, 0x45, 0x00,
	0x20, 0x54, 0x54, 0x7C, 0x54,
	0x7C, 0x0A, 0x09, 0x7F, 0x49,
	0x32, 0x49, 0x49, 0x49, 0x32,
	0x3A, 0x44, 0x44, 0x44, 0x3A, 
	0x32, 0x4A, 0x48, 0x48, 0x30,
	0x3A, 0x41, 0x41, 0x21, 0x7A,
	0x3A, 0x42, 0x40, 0x20, 0x78,
	0x00, 0x9D, 0xA0, 0xA0, 0x7D,
	0x3D, 0x42, 0x42, 0x42, 0x3D, 
	0x3D, 0x40, 0x40, 0x40, 0x3D,
	0x3C, 0x24, 0xFF, 0x24, 0x24,
	0x48, 0x7E, 0x49, 0x43, 0x66,
	0x2B, 0x2F, 0xFC, 0x2F, 0x2B,
	0xFF, 0x09, 0x29, 0xF6, 0x20,
	0xC0, 0x88, 0x7E, 0x09, 0x03,
	0x20, 0x54, 0x54, 0x79, 0x41,
	0x00, 0x00, 0x44, 0x7D, 0x41,
	0x30, 0x48, 0x48, 0x4A, 0x32,
	0x38, 0x40, 0x40, 0x22, 0x7A,
	0x00, 0x7A, 0x0A, 0x0A, 0x72,
	0x7D, 0x0D, 0x19, 0x31, 0x7D,
	0x26, 0x29, 0x29, 0x2F, 0x28,
	0x26, 0x29, 0x29, 0x29, 0x26,
	0x30, 0x48, 0x4D, 0x40, 0x20,
	0x38, 0x08, 0x08, 0x08, 0x08,
	0x08, 0x08, 0x08, 0x08, 0x38,
	0x2F, 0x10, 0xC8, 0xAC, 0xBA,
	0x2F, 0x10, 0x28, 0x34, 0xFA,
	0x00, 0x00, 0x7B, 0x00, 0x00,
	0x08, 0x14, 0x2A, 0x14, 0x22,
	0x22, 0x14, 0x2A, 0x14, 0x08,
	0x55, 0x00, 0x55, 0x00, 0x55, 
	0xAA, 0x55, 0xAA, 0x55, 0xAA, 
	0xFF, 0x55, 0xFF, 0x55, 0xFF, 
	0x00, 0x00, 0x00, 0xFF, 0x00,
	0x10, 0x10, 0x10, 0xFF, 0x00,
	0x14, 0x14, 0x14, 0xFF, 0x00,
	0x10, 0x10, 0xFF, 0x00, 0xFF,
	0x10, 0x10, 0xF0, 0x10, 0xF0,
	0x14, 0x14, 0x14, 0xFC, 0x00,
	0x14, 0x14, 0xF7, 0x00, 0xFF,
	0x00, 0x00, 0xFF, 0x00, 0xFF,
	0x14, 0x14, 0xF4, 0x04, 0xFC,
	0x14, 0x14, 0x17, 0x10, 0x1F,
	0x10, 0x10, 0x1F, 0x10, 0x1F,
	0x14, 0x14, 0x14, 0x1F, 0x00,
	0x10, 0x10, 0x10, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x1F, 0x10,
	0x10, 0x10, 0x10, 0x1F, 0x10,
	0x10, 0x10, 0x10, 0xF0, 0x10,
	0x00, 0x00, 0x00, 0xFF, 0x10,
	0x10, 0x10, 0x10, 0x10, 0x10,
	0x10, 0x10, 0x10, 0xFF, 0x10,
	0x00, 0x00, 0x00, 0xFF, 0x14,
	0x00, 0x00, 0xFF, 0x00, 0xFF,
	0x00, 0x00, 0x1F, 0x10, 0x17,
	0x00, 0x00, 0xFC, 0x04, 0xF4,
	0x14, 0x14, 0x17, 0x10, 0x17,
	0x14, 0x14, 0xF4, 0x04, 0xF4,
	0x00, 0x00, 0xFF, 0x00, 0xF7,
	0x14, 0x14, 0x14, 0x14, 0x14,
	0x14, 0x14, 0xF7, 0x00, 0xF7,
	0x14, 0x14, 0x14, 0x17, 0x14,
	0x10, 0x10, 0x1F, 0x10, 0x1F,
	0x14, 0x14, 0x14, 0xF4, 0x14,
	0x10, 0x10, 0xF0, 0x10, 0xF0,
	0x00, 0x00, 0x1F, 0x10, 0x1F,
	0x00, 0x00, 0x00, 0x1F, 0x14,
	0x00, 0x00, 0x00, 0xFC, 0x14,
	0x00, 0x00, 0xF0, 0x10, 0xF0,
	0x10, 0x10, 0xFF, 0x10, 0xFF,
	0x14, 0x14, 0x14, 0xFF, 0x14,
	0x10, 0x10, 0x10, 0x1F, 0x00,
	0x00, 0x00, 0x00, 0xF0, 0x10,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xFF, 0xFF, 0xFF, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xFF, 0xFF,
	0x0F, 0x0F, 0x0F, 0x0F, 0x0F,
	0x38, 0x44, 0x44, 0x38, 0x44,
	0xFC, 0x4A, 0x4A, 0x4A, 0x34, 
	0x7E, 0x02, 0x02, 0x06, 0x06,
	0x02, 0x7E, 0x02, 0x7E, 0x02,
	0x63, 0x55, 0x49, 0x41, 0x63,
	0x38, 0x44, 0x44, 0x3C, 0x04,
	0x40, 0x7E, 0x20, 0x1E, 0x20,
	0x06, 0x02, 0x7E, 0x02, 0x02,
	0x99, 0xA5, 0xE7, 0xA5, 0x99,
	0x1C, 0x2A, 0x49, 0x2A, 0x1C,
	0x4C, 0x72, 0x01, 0x72, 0x4C,
	0x30, 0x4A, 0x4D, 0x4D, 0x30,
	0x30, 0x48, 0x78, 0x48, 0x30,
	0xBC, 0x62, 0x5A, 0x46, 0x3D,
	0x3E, 0x49, 0x49, 0x49, 0x00,
	0x7E, 0x01, 0x01, 0x01, 0x7E,
	0x2A, 0x2A, 0x2A, 0x2A, 0x2A,
	0x44, 0x44, 0x5F, 0x44, 0x44,
	0x40, 0x51, 0x4A, 0x44, 0x40,
	0x40, 0x44, 0x4A, 0x51, 0x40,
	0x00, 0x00, 0xFF, 0x01, 0x03,
	0xE0, 0x80, 0xFF, 0x00, 0x00,
	0x08, 0x08, 0x6B, 0x6B, 0x08,
	0x36, 0x12, 0x36, 0x24, 0x36,
	0x06, 0x0F, 0x09, 0x0F, 0x06,
	0x00, 0x00, 0x18, 0x18, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x00,
	0x30, 0x40, 0xFF, 0x01, 0x01,
	0x00, 0x1F, 0x01, 0x01, 0x1E,
	0x00, 0x19, 0x1D, 0x17, 0x12,
	0x00, 0x3C, 0x3C, 0x3C, 0x3C,
	0x00, 0x00, 0x00, 0x00, 0x00  
};

// ------------------------
// Send command to display
// ------------------------

void Ssd1306::ssd1306_cmd(uint8_t c) 
{
  uint8_t control = 0x00; 
  Wire.beginTransmission(_i2caddr);
  Wire.write(control);
  Wire.write(c);
  Wire.endTransmission();
}

// ------------------------
// Init the class
// ------------------------

Ssd1306::Ssd1306(uint8_t i2caddr,uint8_t width,uint8_t height,uint8_t orientation) 
{
  _i2caddr = i2caddr;
  _width = width;
  _height = height;
  _orientation = orientation;
}

// ------------------------
// Init the display
// ------------------------

#define SSD1306_SETCONTRAST 0x81
#define SSD1306_DISPLAYALLON_RESUME 0xA4
#define SSD1306_DISPLAYALLON 0xA5
#define SSD1306_NORMALDISPLAY 0xA6
#define SSD1306_INVERTDISPLAY 0xA7
#define SSD1306_DISPLAYOFF 0xAE
#define SSD1306_DISPLAYON 0xAF

#define SSD1306_SETDISPLAYOFFSET 0xD3
#define SSD1306_SETCOMPINS 0xDA

#define SSD1306_SETVCOMDETECT 0xDB

#define SSD1306_SETDISPLAYCLOCKDIV 0xD5
#define SSD1306_SETPRECHARGE 0xD9

#define SSD1306_SETMULTIPLEX 0xA8

#define SSD1306_SETLOWCOLUMN 0x00
#define SSD1306_SETHIGHCOLUMN 0x10

#define SSD1306_SETSTARTLINE 0x40

#define SSD1306_MEMORYMODE 0x20
#define SSD1306_COLUMNADDR 0x21
#define SSD1306_PAGEADDR   0x22

#define SSD1306_COMSCANINC 0xC0
#define SSD1306_COMSCANDEC 0xC8

#define SSD1306_SEGREMAP 0xA0

#define SSD1306_CHARGEPUMP 0x8D

#define SSD1306_EXTERNALVCC 0x1
#define SSD1306_SWITCHCAPVCC 0x2

#define SSD1306_ACTIVATE_SCROLL 0x2F
#define SSD1306_DEACTIVATE_SCROLL 0x2E
#define SSD1306_SET_VERTICAL_SCROLL_AREA 0xA3
#define SSD1306_RIGHT_HORIZONTAL_SCROLL 0x26
#define SSD1306_LEFT_HORIZONTAL_SCROLL 0x27
#define SSD1306_VERTICAL_AND_RIGHT_HORIZONTAL_SCROLL 0x29
#define SSD1306_VERTICAL_AND_LEFT_HORIZONTAL_SCROLL 0x2A


void Ssd1306::begin() 
{
  Wire.begin();
  
  
  // Init sequence
  ssd1306_cmd(0xAE); // Set display OFF		
  ssd1306_cmd(0xD4); // Set Display Clock Divide Ratio / OSC Frequency
  ssd1306_cmd(0x80); // Display Clock Divide Ratio / OSC Frequency 
  ssd1306_cmd(0xA8); // Set Multiplex Ratio
  ssd1306_cmd(_height - 1); // Multiplex Ratio for 128x64 (64-1) else (48 -1)
  ssd1306_cmd(0xD3); // Set Display Offset
  ssd1306_cmd(0x00); // Display Offset
  ssd1306_cmd(0x40); // Set Display Start Line
  ssd1306_cmd(0x8D); // Set Charge Pump
  ssd1306_cmd(0x14); // Charge Pump (0x10 External, 0x14 Internal DC/DC)
  ssd1306_cmd(0x20);         // Addressing mode
  ssd1306_cmd(0x00);
  ssd1306_cmd(0xA1); // Set Segment Re-Map
  ssd1306_cmd(0xC8); // Set Com Output Scan Direction
  ssd1306_cmd(0xDA); // Set COM Hardware Configuration
  if (_height == 32)  
    ssd1306_cmd(0x02); // COM Hardware Configuration
  else  
    ssd1306_cmd(0x12); // COM Hardware Configuration
  ssd1306_cmd(0x81); // Set Contrast
  ssd1306_cmd(0xCF); // Contrast
  ssd1306_cmd(0xD9); // Set Pre-Charge Period
  ssd1306_cmd(0xF1); // Set Pre-Charge Period (0x22 External, 0xF1 Internal)
  ssd1306_cmd(0xDB); // Set VCOMH Deselect Level
  ssd1306_cmd(0x40); // VCOMH Deselect Level
  ssd1306_cmd(0xA4); // Set all pixels OFF
  ssd1306_cmd(0xA6); // Set display not inverted
  ssd1306_cmd(0xAF); // Set display On
}


// ------------------------
// Set a single pixel
// ------------------------

void Ssd1306::oledSetPixel(int16_t x, int16_t y, uint16_t color) 
{
int16_t addr,tmp;

  switch (_orientation) {
  case 0: // Do nothing
          break;
  case 1: tmp = x; x = y; y = tmp;
          x = _width - x - 1;
          break;
  case 2: x = _width - x - 1;
          y = (_height - y) - 1;
          break;
  case 3: tmp = x; x = y; y = tmp;
          y = (_height - y) - 1;
          break;
  }
  addr = x + ((y/8) * _width);
  if (addr < 0 || addr >= BUFFERSIZE) return;
  switch (color) {
     case WHITE:   buffer[addr] |=  (1 << (y&7)); break;
     case BLACK:   buffer[addr] &= ~(1 << (y&7)); break;
     case INVERSE: buffer[addr] ^=  (1 << (y&7)); break;
  }
}


// -------------------------------------------
// Bresenham's algorithm for line
// -------------------------------------------

void Ssd1306::oledDrawLine(int16_t x0, int16_t y0, int16_t x1, int16_t y1, uint16_t color) 
{
int16_t steep = abs(y1 - y0) > abs(x1 - x0);
int16_t tmp;

  if (steep) {
    tmp = x0; x0 = y0; y0 = tmp;
    tmp = x1; x1 = y1; y1 = tmp;
  }

  if (x0 > x1) {
    tmp = x0; x0 = x1; x1 = tmp;
    tmp = y0; y0 = y1; y1 = tmp;
  }

  int16_t dx, dy;
  dx = x1 - x0;
  dy = abs(y1 - y0);

  int16_t err = dx / 2;
  int16_t ystep;

  if (y0 < y1) {
    ystep = 1;
  } else {
    ystep = -1;
  }

  for (; x0<=x1; x0++) {
    if (steep) {
      Ssd1306::oledSetPixel(y0, x0, color);
    } else {
      oledSetPixel(x0, y0, color);
    }
    err -= dy;
    if (err < 0) {
      y0 += ystep;
      err += dx;
    }
  }
}


// -----------------------
// Draw a rectangle
// -----------------------

void Ssd1306::oledDrawRect(int16_t x, int16_t y, int16_t x2, int16_t y2, uint16_t color) 
{
  oledDrawLine(x, y, x2, y, color);
  oledDrawLine(x, y2, x2, y2, color);
  oledDrawLine(x, y, x, y2, color);
  oledDrawLine(x2, y, x2, y2, color);
}


// -----------------------
// Fill a rectangle
// -----------------------

void Ssd1306::oledFillRect(int16_t x, int16_t y, int16_t x2, int16_t y2, uint16_t color) 
{
  if (x >= x2 || y >= y2) return;	

  while (y < y2) {	
    oledDrawLine(x, y, x2, y, color);
    y++;
  }
}

// -----------------------------
// Draw a circle 
// -----------------------------

void Ssd1306::oledDrawCircle(int16_t x0, int16_t y0, int16_t r,uint16_t color) 
{
int16_t f = 1 - r;
int16_t dx = 1;
int16_t dy = -2 * r;
int16_t x = 0;
int16_t y = r;

  oledSetPixel(x0  , y0+r, color);
  oledSetPixel(x0  , y0-r, color);
  oledSetPixel(x0+r, y0  , color);
  oledSetPixel(x0-r, y0  , color);
  while (x<y) {
    if (f >= 0) {
      y--;
      dy = dy + 2;
      f = f + dy;
    }
    x++;
    dx = dx + 2;
    f += dx;
    oledSetPixel(x0 + x, y0 + y, color);
    oledSetPixel(x0 - x, y0 + y, color);
    oledSetPixel(x0 + x, y0 - y, color);
    oledSetPixel(x0 - x, y0 - y, color);
    oledSetPixel(x0 + y, y0 + x, color);
    oledSetPixel(x0 - y, y0 + x, color);
    oledSetPixel(x0 + y, y0 - x, color);
    oledSetPixel(x0 - y, y0 - x, color);
  }
}


// -----------------------------
// Draw a character
// -----------------------------

void Ssd1306::oledDrawChar(int16_t x, int16_t y, unsigned char c, uint16_t color) 
{
uint8_t line;

  for (int8_t i=0; i<6; i++ ) {
    if (i < 5) 
      line = pgm_read_byte(font+(c*5)+i);
    else  
      line = 0x0;
    for (int8_t j=0; j<8; j++, line >>= 1) {
      if (line & 0x1) {
        oledSetPixel(x+i, y+j, color);
      }       
    }
  }
}


// -----------------------------
// Print a line of characters
// -----------------------------

void Ssd1306::oledPrintLine(int16_t x, int16_t y, unsigned char *c, uint16_t color) {

  while (*c != '\0') {
	if (*c == '\n') { c++; x=0; y=y+9; continue; }
	if (*c == '\n') { c++; continue; }   
    oledDrawChar(x,y,*c++,color);
    x = x + 6;
    if (x > (_width - 6)) {
      x = 0;
      y = y + 9;
    }   
  }
}


// ------------------------
// Invert display
// ------------------------

void Ssd1306::oledInvert() 
{
  ssd1306_cmd(SSD1306_INVERTDISPLAY);
}


// ------------------------
// Normal display
// ------------------------

void Ssd1306::oledNormal()
{
  ssd1306_cmd(SSD1306_NORMALDISPLAY);
}


// ------------------------
// Copy buffer to display
// ------------------------

void Ssd1306::oledCopyDisplayBuffer(void) 
{
  ssd1306_cmd(SSD1306_COLUMNADDR);
  if (_width == 64) { 
    ssd1306_cmd(32);  // display starts on row 32
    ssd1306_cmd(95);  // to row 95 
  } else {  
    ssd1306_cmd(0);   // display is on row 0
    ssd1306_cmd(127); // to row 127
  }
  ssd1306_cmd(SSD1306_PAGEADDR);
  ssd1306_cmd(0); 
  ssd1306_cmd((_height / 8) - 1); 

#ifdef __AVR__ 
  // save I2C bitrate
  uint8_t twbrbackup = TWBR;
  TWBR = 12; // set to 400KHz!
#endif
  for (uint16_t i=0; i < ((_width * _height) / 8); i++) {
    Wire.beginTransmission(_i2caddr);
    Wire.write(0x40);
    for (uint8_t x=0; x<16; x++) {
      Wire.write(buffer[i++]);
    }
    i--;
    Wire.endTransmission();
  }
#ifdef __AVR__
  // restore i2C speed
  TWBR = twbrbackup;
#endif
}


// ------------------------
// Clear display buffer
// ------------------------

void Ssd1306::oledClearDisplayBuffer(uint8_t value) 
{
  memset(buffer, value, BUFFERSIZE);
}
