/*
  Arduino-HYT-221-I2C
  ===================

  Output values read from the HYT family via I2C on the Arduino serial console.

  Pinout:
  HYT 221 view from top (Should be the same for all members of the HYT family. Check the Datasheet though.)

     ________
    | =    = |
    |   __   |
    |  /  \  |
    |  \__/  |
    |        |
    |_|_|_|_||
      | | | |
      | | | |
      | | | |
     SDA|VDD|
        |   |
       GND SCL

  Pin 20 en 21

  [Protocol description](http://www.ist-ag.com/eh/ist-ag/resource.nsf/imgref/Download_AHHYT-I2C_E1.2.pdf/$FILE/AHHYT-I2C_E1.2.pdf)
  [Product page](http://www.hygrochip.com/index.php?id=3833&L=1)

*/
void hygroSetup()
{
#define HYT_ADDR 0x28     // I2C address of the HYT 221, 271, 371 and most likely the rest of the family
  Wire.begin();             // Join I2c Bus as master
  pinMode(13, OUTPUT);      // Set pin 13 to output for status LED

}

void loopHygro()
{
  double humidity;
  double temperature;

  Wire.beginTransmission(HYT_ADDR);   // Begin transmission with given device on I2C bus
  Wire.requestFrom(HYT_ADDR, 4);      // Request 4 bytes

  // Read the bytes if they are available
  // The first two bytes are humidity the last two are temperature
  if (Wire.available() == 4) {
    int b1 = Wire.read();
    int b2 = Wire.read();
    int b3 = Wire.read();
    int b4 = Wire.read();

    Wire.endTransmission();           // End transmission and release I2C bus

    // combine humidity bytes and calculate humidity
    int rawHumidity = b1 << 8 | b2;
    // compound bitwise to get 14 bit measurement first two bits
    // are status/stall bit (see intro text)
    rawHumidity =  (rawHumidity &= 0x3FFF);
    humidity = 100.0 / pow(2, 14) * rawHumidity;

    // combine temperature bytes and calculate temperature
    b4 = (b4 >> 2); // Mask away 2 least significant bits see HYT 221 doc
    int rawTemperature = b3 << 6 | b4;
    temperature = 165.0 / pow(2, 14) * rawTemperature - 40;

    Serial.print(humidity);
    Serial.print("% - Temperature: ");
    Serial.println(temperature);

    myGLCD.setColor(VGA_GREEN);
    myGLCD.print("HYT 221", 0, 160);

    myGLCD.setColor(VGA_SILVER);

    myGLCD.print("Vochtigheid:", 0, 170);
    myGLCD.printNumF(humidity , 2, 150, 170);
    myGLCD.print("%", 210, 170);

    myGLCD.print("Temperatuur:", 0 , 180);
    myGLCD.printNumF(temperature , 2, 150, 180);
    myGLCD.print("C", 210, 180);

    _HYT221Hyd = humidity;
    _HYT221Tmp = temperature;
  }
  else {
    Serial.println("Not enough bytes available on Wire.");
  }
  blinkWait();
}

void blinkWait() {
  digitalWrite(13, HIGH);
  delay(1000);
  digitalWrite(13, LOW);
  delay(1000);
}
